<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Blog;
use App\Models\Category;

class BlogController extends Controller
{
 

    public function index()
    {
        $blogs = Blog::all();
        return view('admin.blog.index', compact('blogs'));
    }

    public function create()
    {
        $categories = Category::all();
        return view('admin.blog.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'author' => 'required|string|max:255',
            'category_id' => 'required|exists:categories,id',
            'tags' => 'nullable|string',
            'content' => 'required',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'subheading' => 'nullable|array',
            'long_image' => 'nullable|array',
            'small_images' => 'nullable|array',
            'long_text' => 'nullable|array',
            'buy_now' => 'nullable|array',
        ]);

        $blog = new Blog();
        $blog->title = $request->title;
        $blog->author = $request->author;
        $blog->category_id = $request->category_id;
        $blog->tags = $request->tags;
        $blog->content = $request->content;

        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('blogs', 'public');
            $blog->image = $imagePath;
        }

        $blog->save();

        $sections = [];

        if ($request->has('subheading')) {
            foreach ($request->subheading as $index => $subheading) {
                $section = [
                    'subheading' => $subheading,
                    'long_image' => $request->hasFile("long_image.$index") ? $request->file("long_image.$index")->store('blogs/sections', 'public') : null,
                    'small_images' => $request->hasFile("small_images.$index") ? json_encode(array_map(function ($image) {
                        return $image->store('blogs/sections', 'public');
                    }, $request->file("small_images.$index"))) : null,
                    'long_text' => $request->long_text[$index] ?? null,
                    'buy_now' => $request->buy_now[$index] ?? null,
                ];
                $sections[] = $section;
            }
        }

        $blog->sections()->createMany($sections);

        return redirect()->route('admin.index_blog')->with('success', 'Blog created successfully.');
    }

    public function edit($id)
    {
        $blog = Blog::findOrFail($id);
        $categories = Category::all();

        return view('admin.blog.edit', compact('blog', 'categories'));
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'author' => 'required|string|max:255',
            'category_id' => 'required|exists:categories,id',
            'tags' => 'nullable|string',
            'content' => 'required',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'subheading' => 'nullable|array',
            'long_image' => 'nullable|array',
            'small_images' => 'nullable|array',
            'long_text' => 'nullable|array',
            'buy_now' => 'nullable|array',
        ]);
    
        $blog = Blog::findOrFail($id);
        $blog->title = $request->title;
        $blog->author = $request->author;
        $blog->category_id = $request->category_id;
        $blog->tags = $request->tags;
        $blog->content = $request->content;
    
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('blogs', 'public');
            $blog->image = $imagePath;
        }
    
        $blog->save();
    
        // Handle sections
        $sections = [];
        if ($request->has('subheading')) {
            foreach ($request->subheading as $index => $subheading) {
                $section = [
                    'subheading' => $subheading,
                    'long_image' => $request->hasFile("long_image.$index") ? $request->file("long_image.$index")->store('blogs/sections', 'public') : $blog->sections[$index]->long_image,
                    'small_images' => $request->hasFile("small_images.$index") ? json_encode(array_map(function ($image) {
                        return $image->store('blogs/sections', 'public');
                    }, $request->file("small_images.$index"))) : $blog->sections[$index]->small_images,
                    'long_text' => $request->long_text[$index] ?? $blog->sections[$index]->long_text,
                    'buy_now' => $request->buy_now[$index] ?? $blog->sections[$index]->buy_now,
                ];
                $sections[] = $section;
            }
        }
    
        // Delete old sections
        $blog->sections()->delete();
    
        // Create new sections
        $blog->sections()->createMany($sections);
    
        return redirect()->route('admin.index_blog')->with('success', 'Blog updated successfully.');
    }
    

    public function destroy($id)
    {
        $blog = Blog::findOrFail($id);
        $blog->delete();

        return redirect()->route('admin.index_blog')->with('success', 'Blog deleted successfully.');
    }

    public function shoes()
    {
        // Assuming 'Shoes' is the name of the category
        $category = Category::where('cat', 'Shoes')->first();
        
        if ($category) {
            $blogs = Blog::where('category_id', $category->id)->paginate(10);
            return view('pages.blog.shoes', compact('blogs'));
        } else {
            // Handle case where the category does not exist
            return redirect()->route('index')->with('error', 'Shoes category not found');
        }
    }


    public function allBlogs()
    {
        $blogs = Blog::with('category')->paginate(10); // Adjust pagination as needed
        $latestPosts = Blog::latest()->take(4)->get(); // Fetch the latest posts
        $categories = Category::all(); // Fetch all categories
        return view('pages.blog.all_blogs', compact('blogs', 'latestPosts', 'categories'));
    }
    
    


}
